package gov.va.med.domain.service.messaging.handler;

import gov.va.med.domain.model.IPayload;

/**
 * A IMessageHandler handles a decoded payload (DTO) from an incoming message.  
 * Normal processing is to persist the payload to database tables. 
 * <P>
 * Handlers are configured in the FUNCTIONS table in EVAULT.  A function 
 * knows the bean name of its handler.  
 * <P>
 * Handlers are created only 
 * by the Spring Bean Factory framework using the DefaultApplicationContext
 * and the beans must be defined in the a beans xml in the classpath, 
 * typically in resources/beans/messagingContext.xml.
 * 
 *
 * @author Slava Uchitel
 * @version $$
 * @since MHV 2.0 <br>03/18/2005
 */
public interface IMessageHandler 
{ 
    /**
     * Processes the payload as needed.  Normally, this mean persisting it
     * in a database although it could mean forwarding it to a remote process
     * or any other appropriate handling. 
     * <P>
     * The processing is by default participating in a transaction scoped by 
     * the invoker of this method. So no transaction management is required
     * unless the handler requires its own transaction.
     * <P>
     * The handle method is called as the final step in processing an incoming
     * message.  
     * <P>
     * If an error occurs, the Handler has 4 options:
     * <OL>
     * <LI> throw a RecoverableHandlerException to cause the message to re re-queued
     * for errors that may correct themselves in time such as a 
     * database connection down.
     * <LI> throw an UnrecoverableHandlerException to signal to the 
     * to dump the message to an error table for later investigation
     * <LI>throw a RunTimeException same as UnrecoverableHandlerException.
     * <LI>eat the exception, logging as appropriate.
     * </OL> 
     * 
     * @param payload the DTO resulting from decoding an incoming message
     * @throws RecoverableHandlerException to re-queue the originating message
     * @throws UnrecoverableHandlerException to dump message into an error state. <br>
     * @throws RuntimeException has the same effect as UnrecoverableHandlerException. 
     */
    public abstract void handle(IPayload payload) 
            throws RecoverableHandlerException,
                   UnrecoverableHandlerException;
} 
